﻿// Written by Daniel Cohen Gindi danielgindi@gmail.com

using System ;
using System.Web.UI ;
using System.Web.UI.WebControls ;
using System.ComponentModel ;
using System.Text.RegularExpressions ;
using System.Globalization ;
using System.Security.Permissions ;
using System.Collections;
using System.Drawing;
using System.Collections.Generic;
using System.Text;

/*
 * NOTE:
 *  Some of the configurations need to be dealt with in extra careful,
 *  because they are pure javascript objects (accepting objects, arrays, strings, variable names...)
 *  and some of them accept string or array (automatically encapsulating and escaping string).
 * 
 * Missing an important character at the wrong spot will result in breaking up javascripts,
 * and the CKE won't show up.
 * 
 * */

namespace FredCK.CKEditor
{
	public enum LanguageDirection
	{
		LeftToRight,
		RightToLeft,
        Ui // Inherit from editor UI language direction
    }

    public enum EnterMode
    {
        P,
        BR,
        DIV
    }

    public enum ToolbarLocation
    {
        Top,
        Bottom
    }

    public enum ResizeDirection
    {
        Both,
        Vertical,
        Horizontal
    }

    public enum ScaytMoreSuggestions
    {
        On,
        Off
    }

    public enum DialogButtonsOrder
    {
        OS,
        Rtl,
        Ltr
    }

    /// <summary>
    /// Provides an Web Control for the CKEditor
    /// 
    /// Registration of the control:
    /// On a single page - 
    /// <%@ Register Assembly="FredCK.CKEditor" Namespace="FredCK.CKEditor" TagPrefix="FredCK" %>
    /// Globally, in web.config -
    /// <add tagPrefix="FredCK" namespace="FredCK.CKEditor" assembly="FredCK.CKEditor, Culture=neutral, PublicKeyToken=9ef91de3e191403a" />
    /// </summary>
    [ DefaultProperty("Value") ]
	[ ValidationProperty("Value") ]
    [ ToolboxData("<{0}:CKEditor runat=server></{0}:CKEditor>") ]
    [ Designer("FredCK.CKEditor.CKEditorDesigner")]
    [ ParseChildren(false) ]
	public class CKEditor : System.Web.UI.Control, IPostBackDataHandler
	{
		private bool _IsCompatible ;

        private List<Color> _lstColorButton_Colors = null; // Default value...
        private List<string> _lstRemovePlugins = null; // Default value...
        private List<string> _lstExtraPlugins = null; // Default value...

		public CKEditor()
		{
            // Because the values here are not read from their property gets,
            //   the DefaultValue has no effect and we have to set them in the 
            //   Config array
            SetDefaultValues();
        }

        /// <summary>
        /// Returns the object name of the CKE instance, to be used in javascript.
        /// </summary>
        public string ObjectID
        {
            get { return @"CKEDITOR.instances." + this.ClientID; }
        }

        public void SetDefaultValues()
        {
            Width = Unit.Percentage(100);
            Height = Unit.Pixel(200);
            HtmlEncodeOutput = HtmlEncodeOutput; // Force HtmlEncodeOutput into the configuration array
        }

		#region Base Configurations Properties
		
        /// <summary>
        /// Container of the CKE configuration
        /// </summary>
		[ Browsable( false ) ]
		public CKEditorConfigurations Config
		{
			get 
			{ 
				if ( ViewState["Config"] == null )
                    ViewState["Config"] = new CKEditorConfigurations();
                return (CKEditorConfigurations)ViewState["Config"];
			}
		}

        /// <summary>
        /// The HTML content of the editor.
        /// Interchangeable with Text
        /// 
        /// Default value: empty string
        /// </summary>
        [DefaultValue("")]
		public string Value
		{
			get { object o = ViewState["Value"] ; return ( o == null ? "" : (string)o ) ; }
			set { ViewState["Value"] = value ; }
		}

        /// <summary>
        /// The HTML content of the editor.
        /// Interchangeable with Value
        /// 
        /// Default value: empty string
        /// </summary>
        [DefaultValue("")]
        public string Text
        {
            get { return Value; }
            set { Value = value; }
        }

 		/// <summary>
		/// <p>
		///		Sets or gets the virtual path to the editor's directory. It is
		///		relative to the current page.
		/// </p>
		/// <p>
		///		The default value is "~/ckeditor/".
		/// </p>
		/// <p>
		///		The base path can be also set in the Web.config file using the 
        ///		appSettings section. Just set the "CKEditor:BasePath" for that. 
		///		For example:
		///		<code>
		///		<configuration>
		///			<appSettings>
        ///				<add key="CKEditor:BasePath" value="/scripts/ckeditor/" />
		///			</appSettings>
		///		</configuration>
		///		</code>
		/// </p>
		/// </summary>
		[ DefaultValue( "~/ckeditor/" ) ]
		public string BasePath
		{
			get 
			{ 
				object o = ViewState["BasePath"] ; 

				if ( o == null )
                    o = System.Configuration.ConfigurationSettings.AppSettings["CKEditor:BasePath"];

				return ( o == null ? "~/ckeditor/" : (string)o ) ;
			}
			set { ViewState["BasePath"] = value ; }
		}

		#endregion

		#region Configurations Properties

        #region File Browser Integration

        /// <summary>
        /// The filebrowserBrowseUrl setting is the location of an external file browser, 
        /// that should be launched when "Browse Server" button is pressed.
        /// 
        /// It is also possible to set a separate url for a selected dialog box, 
        /// using the dialog name in file browser settings: filebrowser[dialogName]BrowseUrl
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserBrowseUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserBrowseUrl");
                else this.Config["filebrowserBrowseUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserBrowseUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The filebrowserUploadUrl setting is the location of a script 
        /// that handles file uploads. 
        /// If set, the "Upload" tab will appear in dialog boxes 
        /// (only where such functionality is available, 
        /// i.e. in "Link", "Image" and "Flash" dialog windows).
        /// 
        /// It is also possible to set a separate url for a selected dialog box, 
        /// using the dialog name in file browser settings: filebrowser[dialogName]UploadUrl.
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserUploadUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserUploadUrl");
                else this.Config["filebrowserUploadUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserUploadUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The "features" to use in the file browser popup window.
        /// 
        /// Default value: "location=no,menubar=no,toolbar=no,dependent=yes,minimizable=no,modal=yes,alwaysRaised=yes,resizable=yes,scrollbars=yes"
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FilebrowserWindowFeatures
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserWindowFeatures");
                else this.Config["filebrowserWindowFeatures"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserWindowFeatures"];
                return c == null ? @"location=no,menubar=no,toolbar=no,dependent=yes,minimizable=no,modal=yes,alwaysRaised=yes,resizable=yes,scrollbars=yes" : FromJsString(c);
            }
        }

        /// <summary>
        /// Whether a filler text (non-breaking space entity -  ) will be inserted 
        ///   into empty block elements in HTML output, this is used to render block 
        ///     elements properly with line-height; When a function is instead specified, 
        ///     it'll be passed a CKEDITOR.htmlParser.element to decide whether adding the filler text 
        ///     by expecting a boolean return value. 
        ///     
        /// NOTE: This passes literal text to the config object, which means you can put any value, 
        ///   including a function name, or a function definition. 
        ///   So BE CAREFUL not to break javascript rules!
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public string FillEmptyBlocks
        {
            set { this.Config["fillEmptyBlocks"] = value; }
            get
            {
                string c = this.Config["fillEmptyBlocks"];
                if (c == null) return "true"; // The default value
                return c;
            }
        }

        /// <summary>
        /// The default width of file browser window in CKEditor is set 
        /// to 80% of screen width.
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserWindowWidth
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserWindowWidth");
                else this.Config["filebrowserWindowWidth"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserWindowWidth"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The default hight of file browser window in CKEditor is set 
        /// to 70% of screen height. 
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserWindowHeight
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserWindowHeight");
                else this.Config["filebrowserWindowHeight"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserWindowHeight"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The location of an external file browser, that should be launched when "Browse Server" button is pressed in the Link dialog. 
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserBrowseUrl"/> 
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserLinkBrowseUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserLinkBrowseUrl");
                else this.Config["filebrowserLinkBrowseUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserLinkBrowseUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The location of a script that handles file uploads in the Link dialog.
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserUploadUrl"/>
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserLinkUploadUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserLinkUploadUrl");
                else this.Config["filebrowserLinkUploadUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserLinkUploadUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The default width of file browser window in CKEditor is set 
        /// to 80% of screen width.
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserLinkWindowWidth
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserLinkWindowWidth");
                else this.Config["filebrowserLinkWindowWidth"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserLinkWindowWidth"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The default hight of file browser window in CKEditor is set 
        /// to 70% of screen height. 
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserLinkWindowHeight
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserLinkWindowHeight");
                else this.Config["filebrowserLinkWindowHeight"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserLinkWindowHeight"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The location of an external file browser, that should be launched when "Browse Server" button is pressed in the Image dialog.
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserBrowseUrl"/>
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserImageBrowseUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserImageBrowseUrl");
                else this.Config["filebrowserImageBrowseUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserImageBrowseUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The location of an external file browser, that should be launched when "Browse Server" button is pressed in the Link tab of Image dialog. 
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserBrowseUrl"/> 
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FilebrowserImageBrowseLinkUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserImageBrowseLinkUrl");
                else this.Config["filebrowserImageBrowseLinkUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserImageBrowseLinkUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The location of a script that handles file uploads in the Image dialog.
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserUploadUrl"/>
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserImageUploadUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserImageUploadUrl");
                else this.Config["filebrowserImageUploadUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserImageUploadUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The default width of file browser window in CKEditor is set 
        /// to 80% of screen width.
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserImageWindowWidth
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserImageWindowWidth");
                else this.Config["filebrowserImageWindowWidth"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserImageWindowWidth"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The default hight of file browser window in CKEditor is set 
        /// to 70% of screen height. 
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserImageWindowHeight
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserImageWindowHeight");
                else this.Config["filebrowserImageWindowHeight"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserImageWindowHeight"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The location of an external file browser, that should be launched when "Browse Server" button is pressed in the Flash dialog. 
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserBrowseUrl"/> 
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserFlashBrowseUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserFlashBrowseUrl");
                else this.Config["filebrowserFlashBrowseUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserFlashBrowseUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The location of a script that handles file uploads in the Flash dialog.
        /// If not set, CKEditor will use <see cref="CKEDITOR.config.filebrowserUploadUrl"/>
        /// 
        /// Default value: undefined
        /// 
        /// Set to null to reset to default value
        /// </summary>
        [Category("Configurations")]
        public string FileBrowserFlashUploadUrl
        {
            set
            {
                if (value == null) this.Config.Remove("filebrowserFlashUploadUrl");
                else this.Config["filebrowserFlashUploadUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["filebrowserFlashUploadUrl"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// The default width of file browser window in CKEditor is set 
        /// to 80% of screen width.
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserFlashWindowWidth
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserFlashWindowWidth");
                else this.Config["filebrowserFlashWindowWidth"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserFlashWindowWidth"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// The default hight of file browser window in CKEditor is set 
        /// to 70% of screen height. 
        /// If for some reasons, the default values are not suitable for you, 
        /// you can change it to any other value.
        /// 
        /// To set the size of the window in pixels, 
        /// just set the number value (e.g. "800"). 
        /// If you prefer to set height and width of the window 
        /// in percentage of the screen, remember to add percent 
        /// sign at the end (e.g. "60%").
        /// 
        /// Default value: undefined (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        public Unit FileBrowserFlashWindowHeight
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"filebrowserFlashWindowHeight");
                else this.Config["filebrowserFlashWindowHeight"] = ToJsString(value.ToString());
            }
            get
            {
                string c = this.Config["filebrowserFlashWindowHeight"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        #endregion

        /// <summary>
        /// The maximum height to which the editor can reach using AutoGrow.
        /// Zero means unlimited. 
        /// 
        /// Default value: 0
        /// </summary>
        [Category("Configurations")]
        public int AutoGrow_MaxHeight
        {
            set { this.Config["autoGrow_maxHeight"] = value.ToString(); }
            get 
            {
                string c = this.Config["autoGrow_maxHeight"];
                if (c == null) return 0; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The minimum height to which the editor can reach using AutoGrow.  
        /// 
        /// Default value: 200
        /// </summary>
        [Category("Configurations")]
        public int AutoGrow_MinHeight
        {
            set { this.Config["autoGrow_minHeight"] = value.ToString(); }
            get
            {
                string c = this.Config["autoGrow_minHeight"];
                if (c == null) return 200; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// Whether the replaced element (usually a textarea) is to be 
        /// updated automatically when posting the form containing the editor.
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool AutoUpdateElement
        {
            set { this.Config["autoUpdateElement"] = (value ? "true" : "false"); }
            get 
            {
                string c = this.Config["autoUpdateElement"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The base Z-index for floating dialogs and popups.
        /// 
        /// Default value: 10000
        /// </summary>
        [Category("Configurations")]
        public int BaseFloatZIndex
        {
            set { this.Config["baseFloatZIndex"] = value.ToString(); }
            get 
            {
                string c = this.Config["baseFloatZIndex"];
                if (c == null) return 10000; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The base href URL used to resolve relative and 
        /// absolute URLs in the editor content.
        /// 
        /// Default value: empty string
        /// </summary
        [Category("Configurations")]
        public string BaseHref
        {
            set
            {
                if (value == null) this.Config.Remove("baseHref");
                else this.Config["baseHref"] = ToJsString(value);
            }
            get 
            {
                string c = this.Config["baseHref"];
                return c == null ? string.Empty : FromJsString(c); 
            }
        }

        /// <summary>
        /// A list of keystrokes to be blocked if not defined in the 
        ///  Keystrokes setting. 
        /// In this way it is possible to block the default browser behavior for 
        ///   those keystrokes. 
        /// 
        /// Default value: 
        ///  [
        ///   CKEDITOR.CTRL + 66 /*B*/,
        ///   CKEDITOR.CTRL + 73 /*I*/,
        ///   CKEDITOR.CTRL + 85 /*U*/
        ///  ]
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string BlockedKeystrokes
        {
            set
            {
                if (value == null) this.Config.Remove("blockedKeystrokes");
                else this.Config["blockedKeystrokes"] = value;
            }
            get
            {
                return this.Config["blockedKeystrokes"];
            }
        }

        /// <summary>
        /// Sets the "class" attribute to be used on the body 
        /// element of the editing area.
        /// 
        /// Default value: empty string
        /// </summary>
        [Category("Configurations")]
        public string BodyClass
        {
            set
            {
                if (value == null) this.Config.Remove("bodyClass");
                else this.Config["bodyClass"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["bodyClass"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }
        
        /// <summary>
        /// Sets the "id" attribute to be used on the body 
        /// element of the editing area.
        /// 
        /// Default value: empty string
        /// </summary>
        [Category("Configurations")]
        public string BodyId
        {
            set
            {
                if (value == null) this.Config.Remove("bodyId");
                else this.Config["bodyId"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["bodyId"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// Whether to show the browser native context menu when 
        /// the CTRL or the META (Mac) key is pressed while opening 
        /// the context menu. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool BrowserContextMenuOnCtrl
        {
            set { this.Config["browserContextMenuOnCtrl"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["browserContextMenuOnCtrl"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Holds the style definition to be used to apply the text background color. 
        /// 
        /// Default value: null.
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string ColorButton_BackStyle
        {
            set
            {
                if (value == null) this.Config.Remove("colorButton_backStyle");
                else this.Config["colorButton_backStyle"] = value;
            }
            get
            {
                return this.Config["colorButton_backStyle"];
            }
        }

        /// <summary>
        /// Defines the colors to be displayed in the color selectors.
        /// It's a string containing the hexadecimal notation for HTML colors, 
        ///  without the "#" prefix. 
        /// Since 3.3: A name may be optionally defined by prefixing the 
        ///   entries with the name and the slash character. 
        ///   For example, "FontColor1/FF9900" will be displayed as the 
        ///   color #FF9900 in the selector, but will be outputted as "FontColor1".
        /// 
        /// Default value: '000,800000,8B4513,2F4F4F,008080,000080,4B0082,696969,B22222,A52A2A,DAA520,006400,40E0D0,0000CD,800080,808080,F00,FF8C00,FFD700,008000,0FF,00F,EE82EE,A9A9A9,FFA07A,FFA500,FFFF00,00FF00,AFEEEE,ADD8E6,DDA0DD,D3D3D3,FFF0F5,FAEBD7,FFFFE0,F0FFF0,F0FFFF,F0F8FF,E6E6FA,FFF'
        /// 
        /// * Set null to reset to default value
        /// </summary>
        /// <returns>List of comma separated values, or null when default</returns>
        [Category("Configurations")]
        public string ColorButton_Colors
        {
            set 
            {
                if (value == null) this.Config.Remove("colorButton_colors");
                else this.Config["colorButton_colors"] = ToJsString(value);
                _lstColorButton_Colors = null; // Reset that list
            }
            get 
            {
                string c;
                if (_lstColorButton_Colors != null)
                {
                    c = UpdateConfig_ColorButton_Colors();
                }
                else
                {
                    c = this.Config["colorButton_colors"];
                    if (c != null) c = FromJsString(c); else c = string.Empty;
                }
                return c;
            }
        }

        /// <summary>
        /// Updates the ColorButton_Colors config from its List equivalent.
        /// </summary>
        /// <returns>Config value</returns>
        private string UpdateConfig_ColorButton_Colors()
        {
            if (_lstColorButton_Colors != null)
            {
                StringBuilder sb = new StringBuilder();
                bool first = true;
                foreach (Color c in _lstColorButton_Colors)
                {
                    if (!first) sb.Append(','); else first = false;
                    sb.Append(string.Concat(c.R.ToString(@"x2"), c.G.ToString(@"x2"), c.B.ToString(@"x2")));
                }
                string ret = sb.ToString();
                ColorButton_Colors = ret;
                return ret;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Controls the ColorButton_Colors configuration through a List<Color>
        /// 
        /// * Set null to reset to default value
        /// </summary>
        /// <returns>List of Colors, or null when default</returns>
        public List<Color> ColorButton_Colors_List
        {
            get
            {
                if (_lstColorButton_Colors == null)
                {
                    string[] colors = ColorButton_Colors.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                    _lstColorButton_Colors = new List<Color>(colors.Length);
                    foreach(string c in colors)
                    {
                        _lstColorButton_Colors.Add(System.Drawing.ColorTranslator.FromHtml('#' + c));
                    }
                }
                return _lstColorButton_Colors;
            }
            set
            {
                _lstColorButton_Colors = value;
                if (value == null) ColorButton_Colors = null;
            }
        }

        /// <summary>
        /// Whether to enable the "More Colors..." button in the color selectors.
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool ColorButton_EnableMore
        {
            set { this.Config["colorButton_enableMore"] = (value ? "true" : "false"); }
            get 
            {
                string c = this.Config["colorButton_enableMore"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Holds the style definition to be used to apply the text foreground color. 
        /// 
        /// Default value: null.
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string ColorButton_ForeStyle
        {
            set
            {
                if (value == null) this.Config.Remove("colorButton_foreStyle");
                else this.Config["colorButton_foreStyle"] = value;
            }
            get
            {
                return this.Config["colorButton_foreStyle"];
            }
        }

        /// <summary>
        /// The CSS file(s) to be used to apply style to the contents. 
        /// It should reflect the CSS used in the final pages where the 
        /// contents are to be used.
        /// 
        /// This can be a string or a string[] (Array).
        /// 
        /// Default value: '<CKEditor folder>/contents.css'
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string ContentsCss
        {
            set
            {
                if (value == null) this.Config.Remove("contentsCss");
                else this.Config["contentsCss"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["contentsCss"];
                if (c == null) return null;
                return FromJsString(c);
            } 
        }

        /// <summary>
        /// The writting direction of the language used to write the 
        /// editor contents. 
        /// Allowed values are 'ltr' for Left-To-Right language 
        ///  (like English), or 'rtl' for Right-To-Left languages (like Arabic).
        ///  
        /// Default value: ltr (LanguageDirection.LeftToRight)
        /// </summary>
        [Category("Configurations")]
        public LanguageDirection ContentsLangDirection
        {
            set
            {
                switch (value)
                {
                    default:
                    case LanguageDirection.Ui:
                        this.Config["contentsLangDirection"] = "'ui'";
                        break;
                    case LanguageDirection.LeftToRight:
                        this.Config["contentsLangDirection"] = "'ltr'";
                        break;
                    case LanguageDirection.RightToLeft:
                        this.Config["contentsLangDirection"] = "'rtl'";
                        break;
                }
            }
            get
            {
                switch (this.Config["contentsLangDirection"])
                {
                    default:
                    case "'ui'":
                        return LanguageDirection.Ui;
                    case "'ltr'":
                        return LanguageDirection.LeftToRight;
                    case "'rtl'":
                        return LanguageDirection.RightToLeft;
                }
            }
        }

        /// <summary>
        /// The URL path for the custom configuration file to be loaded. If not overloaded with inline configurations, it defaults to the "config.js" file present in the root of the CKEditor installation directory.
        /// 
        /// CKEditor will recursively load custom configuration files defined inside other custom configuration files.
        /// 
        /// Default value: '<CKEditor folder>/config.js'
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string CustomConfig
        {
            set
            {
                if (value == null) this.Config.Remove("customConfig");
                else this.Config["customConfig"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["customConfig"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// The language to be used if CKEDITOR.config.language is left 
        /// empty and it's not possible to localize the editor to the 
        /// user language.
        /// 
        /// Default value: 'en'
        /// </summary>
        [Category("Configurations")]
        public string DefaultLanguage
        {
            set
            {
                if (value == null) this.Config.Remove("defaultLanguage");
                else this.Config["defaultLanguage"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["defaultLanguage"];
                return c == null ? "en" : FromJsString(c);
            }
        }

        /// <summary>
        /// The color of the dialog background cover. 
        /// It should be a valid CSS color string.
        /// 
        /// Default value: 'white'
        /// </summary>
        [Category("Appearance")]
        [TypeConverter(typeof(WebColorConverter))]
        public Color Dialog_BackgroundCoverColor
        {
            set { this.Config["dialog_backgroundCoverColor"] = ToJsString("#" + value.R.ToString(@"x2") + value.G.ToString(@"x2") + value.B.ToString(@"x2")); }
            get
            {
                string c = this.Config["dialog_backgroundCoverColor"];
                if (c == null) return Color.White; // The default value
                return System.Drawing.ColorTranslator.FromHtml(FromJsString(c));
            }
        }

        /// <summary>
        /// The opacity of the dialog background cover. 
        /// It should be a number within the range [0.0, 1.0]. 
        ///  
        /// Default value: 0.5
        /// </summary>
        [Category("Configurations")]
        public decimal Dialog_BackgroundCoverOpacity
        {
            set { this.Config["dialog_backgroundCoverOpacity"] = value.ToString(@"0.##", new CultureInfo(@"en-US")); }
            get 
            {
                string c = this.Config["dialog_backgroundCoverOpacity"];
                if (c == null) return 0.5m; // The default value
                return decimal.Parse(c);
            }
        }

        /// <summary>
        /// The guildeline to follow when generating the dialog buttons. There are 3 possible options:
        /// 'OS' - the buttons will be displayed in the default order of the user's OS;
        /// 'ltr' - for Left-To-Right order;
        /// 'rtl' - for Right-To-Left order
        ///  
        /// Default value: OS
        /// </summary>
        [Category("Configurations")]
        public DialogButtonsOrder Dialog_ButtonsOrder
        {
            set
            {
                switch (value)
                {
                    default:
                    case DialogButtonsOrder.OS:
                        this.Config["dialog_buttonsOrder"] = "'OS'";
                        break;
                    case DialogButtonsOrder.Rtl:
                        this.Config["dialog_buttonsOrder"] = "'rtl'";
                        break;
                    case DialogButtonsOrder.Ltr:
                        this.Config["dialog_buttonsOrder"] = "'ltr'";
                        break;
                }
            }
            get
            {
                switch (this.Config["dialog_buttonsOrder"])
                {
                    default:
                    case "'OS'":
                        return DialogButtonsOrder.OS;
                    case "'rtl'":
                        return DialogButtonsOrder.Rtl;
                    case "'ltr'":
                        return DialogButtonsOrder.Ltr;
                }
            }
        }

        /// <summary>
        /// The distance of magnetic borders used in moving and 
        /// resizing dialogs, measured in pixels.
        /// 
        /// Default value: 20
        /// </summary>
        [Category("Configurations")]
        public int Dialog_MagnetDistance
        {
            set { this.Config["dialog_magnetDistance"] = value.ToString(); }
            get
            {
                string c = this.Config["dialog_magnetDistance"];
                if (c == null) return 20; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// Disables the built-in spell checker while typing natively 
        /// available in the browser (currently Firefox and Safari only).
        /// 
        /// Even if word suggestions will not appear in 
        /// the CKEditor context menu, this feature is useful to help 
        /// quickly identifying misspelled words.
        /// 
        /// This setting is currently compatible with Firefox only due to limitations in other browsers. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool DisableNativeSpellChecker
        {
            set { this.Config["disableNativeSpellChecker"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["disableNativeSpellChecker"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Disables the "table tools" offered natively by the 
        /// browser (currently Firefox only) to make quick table 
        /// editing operations, like adding or deleting rows and columns. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool DisableNativeTableHandles
        {
            set { this.Config["disableNativeTableHandles"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["disableNativeTableHandles"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Disables the ability of resize objects (image and tables) 
        /// in the editing area.
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool DisableObjectResizing
        {
            set { this.Config["disableObjectResizing"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["disableObjectResizing"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Disables inline styling on read-only elements. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool DisableReadonlyStyling
        {
            set { this.Config["disableReadonlyStyling"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["disableReadonlyStyling"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Sets the doctype to be used when loading the editor content as HTML.
        /// 
        /// Default value: '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">'
        /// </summary>
        [Category("Configurations")]
        public string DocType
        {
            set
            {
                if (value == null) this.Config.Remove("docType");
                else this.Config["docType"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["docType"];
                return c != null ? FromJsString(c) : @"'<!DOCTYPE html PUBLIC ""-//W3C//DTD XHTML 1.0 Transitional//EN"" ""http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"">'";
            }
        }

        /// <summary>
        /// Whether to render or not the editing block area in 
        /// the editor interface. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool EditingBlock
        {
            set { this.Config["editingBlock"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["editingBlock"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The e-mail address anti-spam protection option. The protection will be applied when creating or modifying e-mail links through the editor interface.
        /// 
        /// Two methods of protection can be choosed:
        /// 
        /// * The e-mail parts (name, domain and any other query string) are assembled into a function call pattern. Such function must be provided by the developer in the pages that will use the contents.
        /// * Only the e-mail address is obfuscated into a special string that has no meaning for humans or spam bots, but which is properly rendered and accepted by the browser.
        /// 
        /// Both approaches require JavaScript to be enabled.
        /// 
        /// Recommended values: "encode", "mt(NAME,DOMAIN,SUBJECT,BODY)"
        /// 
        /// Default value: empty string (disabled)
        /// </summary>
        [Category("Configurations")]
        public string EmailProtection
        {
            set
            {
                if (value == null) this.Config.Remove("emailProtection");
                else this.Config["emailProtection"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["emailProtection"];
                return c == null ? string.Empty : FromJsString(c);
            }
        }

        /// <summary>
        /// Allow context-sensitive tab key behaviors, including the following scenarios:
        /// 
        /// When selection is anchored inside table cells:
        ///  * If TAB is pressed, select the contents of the "next" cell.
        ///    If in the last cell in the table,
        ///    add a new row to it and focus its first cell.
        ///  * If SHIFT+TAB is pressed, select the contents of the "previous" cell.
        ///    Do nothing when it's in the first cell.
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool EnableTabKeyTools
        {
            set { this.Config["enableTabKeyTools"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["enableTabKeyTools"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Sets the behavior for the ENTER key. 
        /// It also dictates other behaviour rules in the editor, 
        ///  like whether the <br> element is to be used as a paragraph 
        ///  separator when indenting text. 
        /// The allowed values are the following constants, 
        ///  and their relative behavior:
        /// * CKEDITOR.ENTER_P (1): new <p> paragraphs are created; (EnterMode.P)
        /// * CKEDITOR.ENTER_BR (2): lines are broken with <br> elements; (EnterMode.BR)
        /// * CKEDITOR.ENTER_DIV (3): new <div> blocks are created. (EnterMode.DIV)
        /// 
        /// Note: It's recommended to use the CKEDITOR.ENTER_P value because 
        /// of its semantic value and correctness. 
        /// The editor is optimized for this value.
        /// 
        /// Default value: CKEDITOR.ENTER_P
        /// </summary>
        [Category("Configurations")]
        public EnterMode EnterMode
        {
            set { 
                switch (value)
                {
                    default:
                    case EnterMode.P:
                        this.Config["enterMode"] = "CKEDITOR.ENTER_P";
                        break;
                    case EnterMode.BR:
                        this.Config["enterMode"] = "CKEDITOR.ENTER_BR";
                        break;
                    case EnterMode.DIV:
                        this.Config["enterMode"] = "CKEDITOR.ENTER_DIV";
                        break;
                }                
            }
            get
            {
                string c = this.Config["enterMode"];
                if (c == null) return EnterMode.P;
                switch (c)
                {
                    default:
                    case "CKEDITOR.ENTER_P":
                        return EnterMode.P;
                    case "CKEDITOR.ENTER_BR":
                        return EnterMode.BR;
                    case "CKEDITOR.ENTER_DIV":
                        return EnterMode.DIV;
                }
            }
        }

        /// <summary>
        /// Whether to use HTML entities in the output.
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Entities
        {
            set { this.Config["entities"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["entities"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// An additional list of entities to be used. 
        /// It's a string containing each entry separated by a comma. 
        /// Entities names or number must be used, excluding the "&" prefix 
        /// and the ";" termination.
        /// 
        /// Default value: "#39" // The single quote (') character.
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Entities_additional
        {
            set
            {
                if (value == null) this.Config.Remove("customConfig");
                else this.Config["entities_additional"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["entities_additional"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// Whether to convert some symbols, mathematical symbols, 
        /// and Greek letters to HTML entities. 
        /// This may be more relevant for users typing text written in Greek. 
        /// The list of entities can be found at the 
        /// <a href="http://www.w3.org/TR/html4/sgml/entities.html#h-24.3.1">W3C HTML 4.01 Specification, section 24.3.1.</a>
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Entities_Greek
        {
            set { this.Config["entities_greek"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["entities_greek"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to convert some Latin characters (Latin 
        /// alphabet No. 1, ISO 8859-1) to HTML entities. 
        /// The list of entities can be found at the 
        /// <a href="http://www.w3.org/TR/html4/sgml/entities.html#h-24.2.1">W3C HTML 4.01 Specification, section 24.2.1.</a>
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Entities_Latin
        {
            set { this.Config["entities_latin"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["entities_latin"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to convert all remaining characters, 
        /// not comprised in the ASCII character table, 
        /// to their relative numeric representation of HTML entity. 
        /// For example, the phrase "This is Chinese: 汉语." is outputted 
        /// as "This is Chinese: &#27721;&#35821;." 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool Entities_ProcessNumerical
        {
            set { this.Config["entities_processNumerical"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["entities_processNumerical"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// List of additional plugins to be loaded. 
        /// This is a tool setting which makes it easier to add new plugins, 
        /// without having to touch and possibly breaking 
        /// the CKEDITOR.config.plugins setting.
        /// </summary>
        /// <returns>List of comma separated values, or null when default</returns>
        [Category("Configurations")]
        public string ExtraPlugins
        {
            set
            {
                if (value == null) this.Config.Remove("extraPlugins");
                else this.Config["extraPlugins"] = ToJsString(value);
                _lstExtraPlugins = null; // Reset that list
            }
            get
            {
                string c;
                if (_lstExtraPlugins != null)
                {
                    c = UpdateConfig_ExtraPlugins();
                }
                else
                {
                    c = this.Config["extraPlugins"];
                    if (c != null) c = FromJsString(c); else c = string.Empty;
                }
                return c;
            }
        }

        /// <summary>
        /// Updates the ExtraPlugins config from its List equivalent.
        /// </summary>
        /// <returns>Config value</returns>
        private string UpdateConfig_ExtraPlugins()
        {
            if (_lstExtraPlugins != null)
            {
                StringBuilder sb = new StringBuilder();
                bool first = true;
                foreach (string plugin in _lstExtraPlugins)
                {
                    if (!first) sb.Append(','); else first = false;
                    sb.Append(plugin);
                }
                string ret = sb.ToString();
                ExtraPlugins = ret;
                return ret;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Controls the ExtraPlugins configuration through a List<string>
        /// 
        /// * Set null to reset to default value
        /// </summary>
        /// <returns>List of plugin names, or null when default</returns>
        public List<string> ExtraPlugins_List
        {
            get
            {
                if (_lstExtraPlugins == null)
                {
                    string[] plugins = ExtraPlugins.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                    _lstExtraPlugins = new List<string>(plugins.Length);
                    foreach (string plugin in plugins)
                    {
                        _lstExtraPlugins.Add(plugin);
                    }
                }
                return _lstExtraPlugins;
            }
            set
            {
                _lstExtraPlugins = value;
                if (value == null) ExtraPlugins = null;
            }
        }

        /// <summary>
        /// Add an extra plugin to the ExtraPlugins config
        /// </summary>
        /// <param name="plugin">Plug-in name</param>
        public void AddExtraPlugin(string plugin)
        {
            ExtraPlugins_List.Add(plugin);
        }

        /// <summary>
        /// Removes a plugin from the ExtraPlugins config
        /// </summary>
        /// <param name="plugin">Plug-in name</param>
        public void RemoveExtraPlugin(string plugin)
        {
            ExtraPlugins_List.Remove(plugin);
        }

        /// <summary>
        /// Defines the style to be used to highlight results with the find dialog. 
        /// 
        /// Default value: null.
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Find_Highlight
        {
            set
            {
                if (value == null) this.Config.Remove("find_highlight");
                else this.Config["find_highlight"] = value;
            }
            get
            {
                return this.Config["find_highlight"];
            }
        }

        /// <summary>
        /// The text to be displayed in the Font combo if none of the 
        ///  available values matches the current cursor position or text 
        ///  selection. 
        ///  
        /// Default value: empty string
        /// 
        /// Set to null for default value
        /// </summary>
        [Category("Configurations")]
        public string Font_DefaultLabel
        {
            set
            {
                if (value == null) this.Config.Remove("font_defaultLabel");
                else this.Config["font_defaultLabel"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["font_defaultLabel"];
                return c == null ? "" : FromJsString(c);
            }
        }

        /// <summary>
        /// The list of fonts names to be displayed in the Font combo 
        ///  in the toolbar. 
        /// Entries are separated by semi-colons (;), while it's possible 
        ///  to have more than one font for each entry, in the 
        ///  HTML way (separated by comma). 
        /// A display name may be optionally defined by prefixing 
        ///  the entries with the name and the slash character. 
        /// For example, "Arial/Arial, Helvetica, sans-serif" will be 
        ///  displayed as "Arial" in the list, but will be outputted 
        ///  as "Arial, Helvetica, sans-serif". 
        /// 
        /// Default value: "Arial/Arial, Helvetica, sans-serif;' +
        /// 	            "Comic Sans MS/Comic Sans MS, cursive;" +
        /// 	            "Courier New/Courier New, Courier, monospace;" +
        /// 	            "Georgia/Georgia, serif;" +
        /// 	            "Lucida Sans Unicode/Lucida Sans Unicode, Lucida Grande, sans-serif;" +
        /// 	            "Tahoma/Tahoma, Geneva, sans-serif;" +
        /// 	            "Times New Roman/Times New Roman, Times, serif;" +
        /// 	            "Trebuchet MS/Trebuchet MS, Helvetica, sans-serif;" +
        /// 	            "Verdana/Verdana, Geneva, sans-serif"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Font_Names
        {
            set
            {
                if (value == null) this.Config.Remove("font_names");
                else this.Config["font_names"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["font_names"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// The style definition to be used to apply the font in the text. 
        /// 
        /// Default value:
        ///  {
        ///   element		: 'span',
        ///   styles		: { 'font-family' : '#(family)' },
        ///   overrides	: [ { element : 'font', attributes : { 'face' : null } } ]
        ///  }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Font_Style
        {
            set
            {
                if (value == null) this.Config.Remove("font_style");
                else this.Config["font_style"] = value;
            }
            get
            {
                return this.Config["font_style"];
            }
        }

        /// <summary>
        /// The text to be displayed in the Font Size combo if none of the 
        ///  available values matches the current cursor position or text 
        ///  selection.
        ///  
        /// Default value: empty string
        /// 
        /// Set to null for default value
        /// </summary>
        [Category("Configurations")]
        public string FontSize_DefaultLabel
        {
            set
            {
                if (value == null) this.Config.Remove("fontSize_defaultLabel");
                else this.Config["fontSize_defaultLabel"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["fontSize_defaultLabel"];
                return c == null ? "" : FromJsString(c);
            }
        }

        /// <summary>
        /// The list of fonts size to be displayed in the Font Size combo 
        ///  in the toolbar. Entries are separated by semi-colons (;). 
        /// Any kind of "CSS like" size can be used, like "12px", "2.3em", 
        ///  "130%", "larger" or "x-small". 
        /// A display name may be optionally defined by prefixing the 
        ///  entries with the name and the slash character. 
        /// For example, "Bigger Font/14px" will be displayed as
        ///  "Bigger Font" in the list, but will be outputted as "14px". 
        /// 
        /// Default value: "8/8px;9/9px;10/10px;11/11px;12/12px;14/14px;16/16px;18/18px;20/20px;22/22px;24/24px;26/26px;28/28px;36/36px;48/48px;72/72px"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string FontSize_Sizes
        {
            set
            {
                if (value == null) this.Config.Remove("fontSize_sizes");
                else this.Config["fontSize_sizes"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["fontSize_sizes"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// The style definition to be used to apply the font size in the text. 
        /// 
        /// Default value:
        ///  {
        ///   element		: 'span',
        ///   styles		: { 'font-size' : '#(size)' },
        ///   overrides	: [ { element : 'font', attributes : { 'size' : null } } ]
        ///  }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string FontSize_Style
        {
            set
            {
                if (value == null) this.Config.Remove("fontSize_style");
                else this.Config["fontSize_style"] = value;
            }
            get
            {
                return this.Config["fontSize_style"];
            }
        }

        /// <summary>
        /// Whether to force all pasting operations to insert on plain text 
        /// into the editor, loosing any formatting information possibly 
        /// available in the source text. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool ForcePasteAsPlainText
        {
            set { this.Config["forcePasteAsPlainText"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["forcePasteAsPlainText"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to force using "&" instead of "&amp;" in elements' 
        /// attributes' values.
        /// It's not recommended to change this setting for compliance 
        /// with the W3C XHTML 1.0 standards (C.12, XHTML 1.0). 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool ForceSimpleAmpersand
        {
            set { this.Config["forceSimpleAmpersand"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["forceSimpleAmpersand"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Address" format. 
        /// 
        /// Default value:
        ///  { element : 'address', attributes : { class : 'styledAddress' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_Address
        {
            set
            {
                if (value == null) this.Config.Remove("format_address");
                else this.Config["format_address"] = value;
            }
            get
            {
                return this.Config["format_address"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Normal (DIV)" format. 
        /// 
        /// Default value:
        ///  { element : 'div', attributes : { class : 'normalDiv' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_Div
        {
            set
            {
                if (value == null) this.Config.Remove("format_div");
                else this.Config["format_div"] = value;
            }
            get
            {
                return this.Config["format_div"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 1" format. 
        /// 
        /// Default value:
        ///  { element : 'h1', attributes : { class : 'contentTitle1' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H1
        {
            set
            {
                if (value == null) this.Config.Remove("format_h1");
                else this.Config["format_h1"] = value;
            }
            get
            {
                return this.Config["format_h1"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 2" format. 
        /// 
        /// Default value:
        ///  { element : 'h2', attributes : { class : 'contentTitle2' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H2
        {
            set
            {
                if (value == null) this.Config.Remove("format_h2");
                else this.Config["format_h2"] = value;
            }
            get
            {
                return this.Config["format_h2"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 3" format. 
        /// 
        /// Default value:
        ///  { element : 'h3', attributes : { class : 'contentTitle3' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H3
        {
            set
            {
                if (value == null) this.Config.Remove("format_h3");
                else this.Config["format_h3"] = value;
            }
            get
            {
                return this.Config["format_h3"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 4" format. 
        /// 
        /// Default value:
        ///  { element : 'h4', attributes : { class : 'contentTitle4' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H4
        {
            set
            {
                if (value == null) this.Config.Remove("format_h4");
                else this.Config["format_h4"] = value;
            }
            get
            {
                return this.Config["format_h4"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 5" format. 
        /// 
        /// Default value:
        ///  { element : 'h5', attributes : { class : 'contentTitle5' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H5
        {
            set
            {
                if (value == null) this.Config.Remove("format_h5");
                else this.Config["format_h5"] = value;
            }
            get
            {
                return this.Config["format_h5"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Heading 6" format. 
        /// 
        /// Default value:
        ///  { element : 'h6', attributes : { class : 'contentTitle6' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_H6
        {
            set
            {
                if (value == null) this.Config.Remove("format_h6");
                else this.Config["format_h6"] = value;
            }
            get
            {
                return this.Config["format_h6"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Normal" format. 
        /// 
        /// Default value:
        ///  { element : 'p', attributes : { class : 'normalPara' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_P
        {
            set
            {
                if (value == null) this.Config.Remove("format_p");
                else this.Config["format_p"] = value;
            }
            get
            {
                return this.Config["format_p"];
            }
        }

        /// <summary>
        /// The style definition to be used to apply the "Formatted" format.  
        /// 
        /// Default value:
        ///  { element : 'pre', attributes : { class : 'code' } }
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_Pre
        {
            set
            {
                if (value == null) this.Config.Remove("format_pre");
                else this.Config["format_pre"] = value;
            }
            get
            {
                return this.Config["format_pre"];
            }
        }

        /// <summary>
        /// A list of semi colon separated style names (by default tags) 
        ///  representing the style definition for each entry to be 
        ///  displayed in the Format combo in the toolbar. 
        /// Each entry must have its relative definition configuration 
        ///  in a setting named "format_(tagName)". 
        /// For example, the "p" entry has its definition taken from 
        ///  config.format_p. 
        /// 
        /// Default value: "p;h1;h2;h3;h4;h5;h6;pre;address;div"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Format_Tags
        {
            set
            {
                if (value == null) this.Config.Remove("format_tags");
                else this.Config["format_tags"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["format_tags"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// Indicates whether the contents to be edited are being 
        /// inputted as a full HTML page. 
        /// A full page includes the <html>, <head> and <body> tags. 
        /// The final output will also reflect this setting, 
        /// including the <body> contents only if this setting is disabled.
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool FullPage
        {
            set { this.Config["fullPage"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["fullPage"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The height of editing area( content ), in relative or absolute, 
        /// e.g. 30px, 5em. 
        /// 
        /// Note: Percentage unit is not supported yet. e.g. 30%.
        /// 
        /// Default value: "200"
        /// </summary>
        [Category("Configurations")]
        [DefaultValue("200px")]
        public Unit Height
        {
            set 
            {
                if (value.IsEmpty) this.Config.Remove(@"height");
                else this.Config["height"] = ToJsString(value.ToString()); 
            }
            get { return Unit.Parse(FromJsString(this.Config["height"] ?? "200px")); }
        }

        /// <summary>
        /// Whether escape HTML when editor update original input element. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        [DefaultValue(true)]
        public bool HtmlEncodeOutput
        {
            set { this.Config["htmlEncodeOutput"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["htmlEncodeOutput"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether the editor must output an empty value ("") if
        /// it's contents is made by an empty paragraph only. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool IgnoreEmptyParagraph
        {
            set { this.Config["ignoreEmptyParagraph"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["ignoreEmptyParagraph"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to remove links when emptying the link URL field 
        /// in the image dialog. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Image_RemoveLinkByEmptyURL
        {
            set { this.Config["image_removeLinkByEmptyURL"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["image_removeLinkByEmptyURL"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// A list associating keystrokes to editor commands. 
        /// Each element in the list is an array where the first item is 
        ///   the keystroke, and the second is the name of the command to be executed. 
        /// 
        /// Default value: 
        ///  [
        ///   [ CKEDITOR.ALT + 121 /*F10*/, 'toolbarFocus' ],
        ///   [ CKEDITOR.ALT + 122 /*F11*/, 'elementsPathFocus' ],
        ///   [ CKEDITOR.SHIFT + 121 /*F10*/, 'contextMenu' ],
        ///   [ CKEDITOR.CTRL + 90 /*Z*/, 'undo' ],
        ///   [ CKEDITOR.CTRL + 89 /*Y*/, 'redo' ],
        ///   [ CKEDITOR.CTRL + CKEDITOR.SHIFT + 90 /*Z*/, 'redo' ],
        ///   [ CKEDITOR.CTRL + 76 /*L*/, 'link' ],
        ///   [ CKEDITOR.CTRL + 66 /*B*/, 'bold' ],
        ///   [ CKEDITOR.CTRL + 73 /*I*/, 'italic' ],
        ///   [ CKEDITOR.CTRL + 85 /*U*/, 'underline' ],
        ///   [ CKEDITOR.ALT + 109 /*-*/, 'toolbarCollapse' ]
        ///  ]
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Keystrokes
        {
            set
            {
                if (value == null) this.Config.Remove("keystrokes");
                else this.Config["keystrokes"] = value;
            }
            get
            {
                return this.Config["keystrokes"];
            }
        }

        /// <summary>
        /// The user interface language localization to use. 
        /// If empty, the editor automatically localize the editor to the 
        /// user language, if supported, otherwise the 
        /// CKEDITOR.config.defaultLanguage language is used.
        /// 
        /// Default value: empty string
        /// </summary>
        [Category("Configurations")]
        public string Language
        {
            set
            {
                if (value == null) this.Config.Remove("language");
                else this.Config["language"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["language"];
                return c == null ? "" : FromJsString(c);
            }
        }

        /// <summary>
        /// A comma separated list of items group names to be 
        ///  displayed in the context menu. 
        /// The items order will reflect the order in this list 
        ///  if no priority has been defined in the groups.
        /// 
        /// Default value: "clipboard,form,tablecell,tablecellproperties,tablerow,tablecolumn,table,anchor,link,image,flash,checkbox,radio,textfield,hiddenfield,imagebutton,button,select,textarea"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Menu_Groups
        {
            set
            {
                if (value == null) this.Config.Remove("menu_groups");
                else this.Config["menu_groups"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["menu_groups"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// The amount of time, in milliseconds, the editor waits before 
        /// showing submenu options when moving the mouse over options 
        /// that contains submenus, like the "Cell Properties" entry for tables. 
        /// 
        /// Default value: 400
        /// </summary>
        [Category("Configurations")]
        public int Menu_SubMenuDelay
        {
            set { this.Config["menu_subMenuDelay"] = value.ToString(); }
            get
            {
                string c = this.Config["menu_subMenuDelay"];
                if (c == null) return 400; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The HTML to load in the editor when the "new page" command 
        /// is executed. 
        /// 
        /// Default value: empty string
        /// </summary>
        [Category("Configurations")]
        public string Newpage_Html
        {
            set
            {
                if (value == null) this.Config.Remove("newpage_html");
                else this.Config["newpage_html"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["newpage_html"];
                return c == null ? "" : FromJsString(c);
            }
        }

        /// <summary>
        /// The file that provides the MS Word cleanup function for 
        /// pasting operations. 
        /// Note: This is a global configuration shared by all editor instances 
        /// present in the page. 
        /// 
        /// Default value: "default"
        /// </summary>
        [Category("Configurations")]
        public string PasteFromWordCleanupFile
        {
            set
            {
                if (value == null) this.Config.Remove("pasteFromWordCleanupFile");
                else this.Config["pasteFromWordCleanupFile"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["pasteFromWordCleanupFile"];
                return c == null ? "default" : FromJsString(c);
            }
        }

        /// <summary>
        /// Whether to transform MS Word outline numbered headings into lists.
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool PasteFromWordNumberedHeadingToList
        {
            set { this.Config["pasteFromWordNumberedHeadingToList"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["pasteFromWordNumberedHeadingToList"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to prompt the user about the clean up of content being pasted from MS Word. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool PasteFromWordPromptCleanup
        {
            set { this.Config["pasteFromWordPromptCleanup"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["pasteFromWordPromptCleanup"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to ignore all font related formatting styles, including:
        /// * font size;
        /// * font family;
        /// * font foreground/background color.
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool PasteFromWordRemoveFontStyles
        {
            set { this.Config["pasteFromWordRemoveFontStyles"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["pasteFromWordRemoveFontStyles"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Whether to remove element styles that can't be 
        ///  managed with the editor.
        /// Note that this doesn't handle the font specific styles, 
        ///  which depends on the pasteFromWordRemoveFontStyles setting instead. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool PasteFromWordRemoveStyles
        {
            set { this.Config["pasteFromWordRemoveStyles"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["pasteFromWordRemoveStyles"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Comma separated list of plugins to load and initialize for 
        ///  an editor instance. 
        /// This should be rarely changed, using instead the 
        ///  CKEDITOR.config.extraPlugins and CKEDITOR.config.removePlugins 
        ///  for customizations.
        /// 
        /// Default value: empty array
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string[] Plugins
        {
            set
            {
                if (value != null)
                {
                    StringBuilder sb = new StringBuilder();
                    bool comma = false;
                    sb.Append('[');
                    foreach (string css in value)
                    {
                        if (comma) sb.Append(@","); else comma = true;
                        sb.Append(ToJsString(css));
                    }
                    sb.Append(']');
                    this.Config["plugins"] = sb.ToString();
                    ViewState["Plugins"] = value;
                }
                else
                {
                    this.Config.Remove("plugins");
                    ViewState.Remove("Plugins");
                }
            }
            get
            {
                return (string[])ViewState["Plugins"];
            }
        }

        /// <summary>
        /// List of regular expressions to be executed over the input HTML, 
        ///   indicating code that must stay untouched.
        /// 
        /// Default value: [] (empty array)
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        [Category("Configurations")]
        public string ProtectedSource
        {
            set
            {
                if (value == null) this.Config.Remove("protectedSource");
                else this.Config["protectedSource"] = value;
            }
            get
            {
                return this.Config["protectedSource"] ?? "[]";
            }
        }
        
        /// <summary>
        /// The dialog contents to removed. 
        /// It's a string composed by dialog name and tab name with a colon between them. 
        /// Separate each pair with semicolon (see example). 
        /// 
        /// Note: All names are case-sensitive. 
        /// Note: Be cautious when specifying dialog tabs that are mandatory, like "info", 
        ///       dialog functionality might be broken because of this! 
        /// 
        /// Example: config.removeDialogTabs = 'flash:advanced;image:Link';
        /// 
        /// Set to null for default value
        /// 
        /// Default value: empty string
        /// </summary>
        [Category("Configurations")]
        public string RemoveDialogTabs
        {
            set
            {
                if (value == null) this.Config.Remove("removeDialogTabs");
                else this.Config["removeDialogTabs"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["removeDialogTabs"];
                if (c == null) return @"";
                return FromJsString(c);
            }
        }

        /// <summary>
        /// A comma separated list of elements attributes to be 
        /// removed when executing the "remove format" command. 
        /// 
        /// Default value: "class,style,lang,width,height,align,hspace,valign"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string RemoveFormatAttributes
        {
            set
            {
                if (value == null) this.Config.Remove("removeFormatAttributes");
                else this.Config["removeFormatAttributes"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["removeFormatAttributes"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// A comma separated list of elements to be removed when executing 
        /// the "remove format" command. 
        /// Note that only inline elements are allowed. 
        /// 
        /// Default value: "b,big,code,del,dfn,em,font,i,ins,kbd,q,samp,small,span,strike,strong,sub,sup,tt,u,var"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string RemoveFormatTags
        {
            set
            {
                if (value == null) this.Config.Remove("removeFormatTags");
                else this.Config["removeFormatTags"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["removeFormatTags"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// List of plugins that must not be loaded.
        /// This is a tool setting which makes it easier to avoid loading plugins definied in the CKEDITOR.config.plugins setting, 
        /// without having to touch it and potentially breaking it.
        /// </summary>
        /// <returns>List of comma separated values, or null when default</returns>
        [Category("Configurations")]
        public string RemovePlugins
        {
            set 
            {
                if (value == null) this.Config.Remove("removePlugins");
                else this.Config["removePlugins"] = ToJsString(value);
                _lstRemovePlugins = null; // Reset that list
            }
            get
            {
                string c;
                if (_lstRemovePlugins != null)
                {
                    c = UpdateConfig_RemovePlugins();
                }
                else
                {
                    c = this.Config["removePlugins"];
                    if (c != null) c = FromJsString(c); else c = string.Empty;
                }
                return c;
            }
        }

        /// <summary>
        /// Updates the RemovePlugins config from its List equivalent.
        /// </summary>
        /// <returns>Config value</returns>
        private string UpdateConfig_RemovePlugins()
        {
            if (_lstRemovePlugins != null)
            {
                StringBuilder sb = new StringBuilder();
                bool first = true;
                foreach (string plugin in _lstRemovePlugins)
                {
                    if (!first) sb.Append(','); else first = false;
                    sb.Append(plugin);
                }
                string ret = sb.ToString();
                RemovePlugins = ret;
                return ret;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Controls the RemovePlugins configuration through a List<string>
        /// 
        /// * Set null to reset to default value
        /// </summary>
        /// <returns>List of plugin names, or null when default</returns>
        public List<string> RemovePlugins_List
        {
            get
            {
                if (_lstRemovePlugins == null)
                {
                    string[] plugins = RemovePlugins.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                    _lstRemovePlugins = new List<string>(plugins.Length);
                    foreach(string plugin in plugins)
                    {
                        _lstRemovePlugins.Add(plugin);
                    }
                }
                return _lstRemovePlugins;
            }
            set
            {
                _lstRemovePlugins = value;
                if (value == null) RemovePlugins = null;
            }
        }

        /// <summary>
        /// Removes a plugin using the RemovePlugins config
        /// </summary>
        /// <param name="plugin">Plug-in name to remove</param>
        public void RemovePlugin(string plugin)
        {
            RemovePlugins_List.Add(plugin);
        }

        /// <summary>
        /// The directions where resizing is enabled. 
        /// It can be 'both', 'vertical' or 'horizontal' 
        ///  
        /// Default value: Both
        /// </summary>
        [Category("Configurations")]
        public ResizeDirection ResizeDir
        {
            set
            {
                switch (value)
                {
                    default:
                    case ResizeDirection.Both:
                        this.Config["resize_dir"] = "'both'";
                        break;
                    case ResizeDirection.Vertical:
                        this.Config["resize_dir"] = "'vertical'";
                        break;
                    case ResizeDirection.Horizontal:
                        this.Config["resize_dir"] = "'horizontal'";
                        break;
                }
            }
            get
            {
                switch (this.Config["resize_dir"])
                {
                    default:
                    case "'both'":
                        return ResizeDirection.Both;
                    case "'vertical'":
                        return ResizeDirection.Vertical;
                    case "'horizontal'":
                        return ResizeDirection.Horizontal;
                }
            }
        }

        /// <summary>
        /// Whether to enable the resizing feature. 
        /// If disabled the resize handler will not be visible. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Resize_Enabled
        {
            set { this.Config["resize_enabled"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["resize_enabled"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The maximum editor height, in pixels, when resizing it with 
        /// the resize handle. 
        /// 
        /// Default value: 3000
        /// </summary>
        [Category("Configurations")]
        public int Resize_MaxHeight
        {
            set { this.Config["resize_maxHeight"] = value.ToString(); }
            get
            {
                string c = this.Config["resize_maxHeight"];
                if (c == null) return 3000; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The maximum editor width, in pixels, when resizing it 
        /// with the resize handle. 
        /// 
        /// Default value: 3000
        /// </summary>
        [Category("Configurations")]
        public int Resize_MaxWidth
        {
            set { this.Config["resize_maxWidth"] = value.ToString(); }
            get
            {
                string c = this.Config["resize_maxWidth"];
                if (c == null) return 3000; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The minimum editor height, in pixels, when resizing it with 
        /// the resize handle. 
        /// 
        /// Default value: 250
        /// </summary>
        [Category("Configurations")]
        public int Resize_MinHeight
        {
            set { this.Config["resize_minHeight"] = value.ToString(); }
            get
            {
                string c = this.Config["resize_minHeight"];
                if (c == null) return 250; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The minimum editor width, in pixels, when resizing it 
        /// with the resize handle. 
        /// 
        /// Default value: 750
        /// </summary>
        [Category("Configurations")]
        public int Resize_MinWidth
        {
            set { this.Config["resize_minWidth"] = value.ToString(); }
            get
            {
                string c = this.Config["resize_minWidth"];
                if (c == null) return 750; // The default value
                return int.Parse(c);
            }
        }

        #region SCAYT configurations, since 3.3
        /// <summary>
        /// If enabled (true), turns on SCAYT automatically after loading the editor. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Scayt_AutoStartup
        {
            set { this.Config["scayt_autoStartup"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["scayt_autoStartup"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Customizes the display of SCAYT context menu commands 
        ///   ("Add Word", "Ignore" and "Ignore All"). 
        /// It must be a string with one or more of the following words 
        ///   separated by a pipe ("|"):
        /// 
        /// "off": disables all options.
        /// "all": enables all options.
        /// "ignore": enables the "Ignore" option.
        /// "ignoreall": enables the "Ignore All" option.
        /// "add": enables the "Add Word" option.
        /// 
        /// Default value: "all"
        /// </summary>
        [Category("Configurations")]
        public string Scayt_ContextCommands
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_contextCommands");
                else this.Config["scayt_contextCommands"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_contextCommands"];
                return c == null ? @"all" : FromJsString(c);
            }
        }

        /// <summary>
        /// Define order of placing of SCAYT context menu items by groups. 
        /// It must be a string with one or more of the following words 
        ///   separated by a pipe ("|"):
        /// 
        /// 'suggest' - main suggestion word list,
        /// 'moresuggest' - more suggestions word list,
        /// 'control' - SCAYT commands, such as 'Ignore' and 'Add Word'
        /// 
        /// Default value: "suggest|moresuggest|control"
        /// </summary>
        [Category("Configurations")]
        public string Scayt_ContextMenuItemsOrder
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_contextMenuItemsOrder");
                else this.Config["scayt_contextMenuItemsOrder"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_contextMenuItemsOrder"];
                return c == null ? @"suggest|moresuggest|control" : FromJsString(c);
            }
        }

        /// <summary>
        /// Makes it possible to place the SCAYT context menu items above others. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool Scayt_ContextMenuOntop 
        {
            set { this.Config["scayt_contextMenuOntop "] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["scayt_contextMenuOntop "];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// Links SCAYT to custom dictionaries. 
        /// It's a string containing dictionary ids separared by commas (","). 
        /// Available only for licensed version. 
        /// Further details at http://wiki.spellchecker.net/doku.php?id=custom_dictionary_support . 
        /// 
        /// Example: Scayt_CustomDictionaryIds = "3021,3456,3478";
        /// 
        /// Default value: ""
        /// </summary>
        [Category("Configurations")]
        public string Scayt_CustomDictionaryIds
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_customDictionaryIds");
                else this.Config["scayt_customDictionaryIds"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_customDictionaryIds"];
                return c == null ? @"" : FromJsString(c);
            }
        }

        /// <summary>
        /// Sets the customer ID for SCAYT. 
        /// Required for migration from free version with banner to paid version. 
        /// Further details at http://wiki.spellchecker.net/doku.php?id=custom_dictionary_support . 
        /// 
        /// Default value: ""
        /// </summary>
        [Category("Configurations")]
        public string Scayt_Customerid
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_customerid");
                else this.Config["scayt_customerid"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_customerid"];
                return c == null ? @"" : FromJsString(c);
            }
        }

        /// <summary>
        /// Defines the number of SCAYT suggestions to show in the main context menu. 
        /// The possible values are:
        ///   0 (zero): All suggestions are displayed in the main context menu.
        ///   Positive number: The maximum number of suggestions to shown in context menu. Other entries will be shown in "More Suggestions" sub-menu.
        ///   Negative number: No suggestions are shown in the main context menu. All entries will be listed in the "Suggestions" sub-menu.
        /// 
        /// Default value: 5
        /// </summary>
        [Category("Configurations")]
        public int Scayt_MaxSuggestions
        {
            set { this.Config["scayt_maxSuggestions"] = value.ToString(); }
            get
            {
                string c = this.Config["scayt_maxSuggestions"];
                if (c == null) return 5; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// Enables/disables the "More Suggestions" sub-menu in the context menu. 
        /// The possible values are "on" or "off". 
        ///  
        /// Default value: On
        /// </summary>
        [Category("Configurations")]
        public ScaytMoreSuggestions Scayt_MoreSuggestions
        {
            set
            {
                switch (value)
                {
                    default:
                    case ScaytMoreSuggestions.On:
                        this.Config["scayt_moreSuggestions"] = "'on'";
                        break;
                    case ScaytMoreSuggestions.Off:
                        this.Config["scayt_moreSuggestions"] = "'vertical'";
                        break;
                }
            }
            get
            {
                switch (this.Config["scayt_moreSuggestions"])
                {
                    default:
                    case "'on'":
                        return ScaytMoreSuggestions.On;
                    case "'off'":
                        return ScaytMoreSuggestions.Off;
                }
            }
        }

        /// <summary>
        /// Sets the default spellchecking language for SCAYT. 
        /// 
        /// Default value: "en_US"
        /// </summary>
        [Category("Configurations")]
        public string Scayt_SLang
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_sLang");
                else this.Config["scayt_sLang"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_sLang"];
                return c == null ? @"en_US" : FromJsString(c);
            }
        }

        /// <summary>
        /// Set the URL to SCAYT core. 
        /// Required to switch to licensed version of SCAYT application. 
        /// Further details at http://wiki.spellchecker.net/doku.php?id=3rd:wysiwyg:fckeditor:wscckf3l . 
        /// 
        /// Default value: ""
        /// </summary>
        [Category("Configurations")]
        public string Scayt_SrcUrl
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_srcUrl");
                else this.Config["scayt_srcUrl"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_srcUrl"];
                return c == null ? @"" : FromJsString(c);
            }
        }

        /// <summary>
        /// Sets the visibility of the SCAYT tabs in the settings dialog and 
        ///   toolbar button. 
        /// The value must contain a "1" (enabled) or "0" (disabled) number 
        ///   for each of the following entries, in this precise order, 
        ///   separated by a comma (","): 
        ///   "Options", "Languages" and "Dictionary". 
        /// 
        /// Default value: "1,1,1"
        /// </summary>
        [Category("Configurations")]
        public string Scayt_UiTabs
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_uiTabs");
                else this.Config["scayt_uiTabs"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_uiTabs"];
                return c == null ? @"1,1,1" : FromJsString(c);
            }
        }

        /// <summary>
        /// Makes it possible to activate a custom dictionary on SCAYT. 
        /// The user dictionary name must be used. 
        /// Available only for licensed version. 
        /// 
        /// Default value: ""
        /// </summary>
        [Category("Configurations")]
        public string Scayt_UserDictionaryName
        {
            set
            {
                if (value == null) this.Config.Remove("scayt_userDictionaryName");
                else this.Config["scayt_userDictionaryName"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["scayt_userDictionaryName"];
                return c == null ? @"" : FromJsString(c);
            }
        }

        #endregion

        [Category("Configurations")]
        [TypeConverter(typeof(WebControl)), Themeable(false), DefaultValue(""), IDReferenceProperty]
        public string SharedSpacesTop
        {
            get
            {
                object o = this.ViewState["SharedSpacesTop"];
                if (o != null) return (string)o;
                return string.Empty;
            }
            set
            {
                this.ViewState["SharedSpacesTop"] = value;
            }
        }

        [Category("Configurations")]
        [Themeable(false), DefaultValue("")]
        public string SharedSpacesTopClientID
        {
            get
            {
                object o = this.ViewState["SharedSpacesTopClientID"];
                if (o != null) return (string)o;
                return string.Empty;
            }
            set
            {
                this.ViewState["SharedSpacesTopClientID"] = value;
            }
        }

        [Category("Configurations")]
        [TypeConverter(typeof(WebControl)), Themeable(false), DefaultValue(""), IDReferenceProperty]
        public string SharedSpacesBottom
        {
            get
            {
                object o = this.ViewState["SharedSpacesBottom"];
                if (o != null) return (string)o;
                return string.Empty;
            }
            set
            {
                this.ViewState["SharedSpacesBottom"] = value;
            }
        }

        [Category("Configurations")]
        [Themeable(false), DefaultValue("")]
        public string SharedSpacesBottomClientID
        {
            get
            {
                object o = this.ViewState["SharedSpacesBottomClientID"];
                if (o != null) return (string)o;
                return string.Empty;
            }
            set
            {
                this.ViewState["SharedSpacesBottomClientID"] = value;
            }
        }

        /// <summary>
        /// Just like the CKEDITOR.config.enterMode setting, 
        ///  it defines the behavior for the SHIFT+ENTER key. 
        /// The allowed values are the following constants, 
        ///  and their relative behavior:
        /// * CKEDITOR.ENTER_P (1): new <p> paragraphs are created; (EnterMode.P)
        /// * CKEDITOR.ENTER_BR (2): lines are broken with <br> elements; (EnterMode.BR)
        /// * CKEDITOR.ENTER_DIV (3): new <div> blocks are created. (EnterMode.DIV)
        /// 
        /// Default value: CKEDITOR.ENTER_BR
        /// </summary>
        [Category("Configurations")]
        public EnterMode ShiftEnterMode
        {
            set
            {
                switch (value)
                {
                    case EnterMode.P:
                        this.Config["shiftEnterMode"] = "CKEDITOR.ENTER_P";
                        break;
                    default:
                    case EnterMode.BR:
                        this.Config["shiftEnterMode"] = "CKEDITOR.ENTER_BR";
                        break;
                    case EnterMode.DIV:
                        this.Config["shiftEnterMode"] = "CKEDITOR.ENTER_DIV";
                        break;
                }
            }
            get
            {
                string c = this.Config["shiftEnterMode"];
                if (c == null) return EnterMode.P;
                switch (c)
                {
                    case "CKEDITOR.ENTER_P":
                        return EnterMode.P;
                    default:
                    case "CKEDITOR.ENTER_BR":
                        return EnterMode.BR;
                    case "CKEDITOR.ENTER_DIV":
                        return EnterMode.DIV;
                }
            }
        }

        /// <summary>
        /// The skin to load. 
        /// It may be the name of the skin folder inside the 
        ///  editor installation path, or the name and the path 
        ///  separated by a comma.
        /// 
        /// Default value: "default"
        /// 
        /// Set to null for default value
        /// </summary>
        [Category("Configurations")]
        public string Skin
        {
            set
            {
                if (value == null) this.Config.Remove("skin");
                else this.Config["skin"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["skin"];
                return c == null ? @"default" : FromJsString(c);
            }
        }

        /// <summary>
        /// The number of columns to be generated by the smilies matrix. 
        /// 
        /// Default value: 8
        /// </summary>
        [Category("Configurations")]
        public int Smiley_Columns
        {
            set { this.Config["smiley_columns"] = value.ToString(); }
            get
            {
                string c = this.Config["smiley_columns"];
                if (c == null) return 8; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The description to be used for each of the smilies 
        ///  defined in the CKEDITOR.config.smiley_images setting. 
        /// Each entry in this array list must match its relative pair 
        ///  in the CKEDITOR.config.smiley_images setting.  
        /// 
        /// Default value: 
        /// {":)", ":(", ";)", ":D", ":/", ":P",
        /// "", "", "", "", "", "",
        /// "", ";(", "", "", "", "",
        /// "", ":kiss", "" }
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string[] Smiley_Descriptions
        {
            set
            {
                if (value != null)
                {
                    StringBuilder sb = new StringBuilder();
                    bool comma = false;
                    sb.Append('[');
                    foreach (string css in value)
                    {
                        if (comma) sb.Append(@","); else comma = true;
                        sb.Append(ToJsString(css));
                    }
                    sb.Append(']');
                    this.Config["smiley_descriptions"] = sb.ToString();
                    ViewState["Smiley_Descriptions"] = value;
                }
                else
                {
                    this.Config.Remove("smiley_descriptions");
                    ViewState.Remove("Smiley_Descriptions");
                }
            }
            get
            {
                return (string[])ViewState["Smiley_Descriptions"];
            }
        }

        /// <summary>
        /// The file names for the smileys to be displayed. 
        /// These files must be contained inside the URL path defined 
        ///  with the CKEDITOR.config.smiley_path setting. 
        /// 
        /// Default value: 
        /// { "regular_smile.gif","sad_smile.gif","wink_smile.gif",
        /// "teeth_smile.gif","confused_smile.gif","tounge_smile.gif",
        /// "embaressed_smile.gif","omg_smile.gif","whatchutalkingabout_smile.gif",
        /// "angry_smile.gif","angel_smile.gif","shades_smile.gif",
        /// "devil_smile.gif","cry_smile.gif","lightbulb.gif","thumbs_down.gif",
        /// "thumbs_up.gif","heart.gif","broken_heart.gif","kiss.gif",
        /// "envelope.gif" }
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string[] Smiley_Images
        {
            set
            {
                if (value != null)
                {
                    StringBuilder sb = new StringBuilder();
                    bool comma = false;
                    sb.Append('[');
                    foreach (string css in value)
                    {
                        if (comma) sb.Append(@","); else comma = true;
                        sb.Append(ToJsString(css));
                    }
                    sb.Append(']');
                    this.Config["smiley_images"] = sb.ToString();
                    ViewState["Smiley_Images"] = value;
                }
                else
                {
                    this.Config.Remove("smiley_images");
                    ViewState.Remove("Smiley_Images");
                }
            }
            get
            {
                return (string[])ViewState["Smiley_Images"];
            }
        }

        /// <summary>
        /// The base path used to build the URL for the smiley images. 
        /// It must end with a slash. 
        /// 
        /// Default value: + "plugins/smiley/images/"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Smiley_Path
        {
            set
            {
                if (value == null) this.Config.Remove("smiley_path");
                else this.Config["smiley_path"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["smiley_path"];
                if (c == null) return null;
                return FromJsString(c);
            }
        }

        /// <summary>
        /// Sets whether the editor should have the focus when the page loads. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool StartupFocus
        {
            set { this.Config["startupFocus"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["startupFocus"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The mode to load at the editor startup. 
        /// It depends on the plugins loaded. 
        /// By default, the "wysiwyg" and "source" modes are available. 
        /// 
        /// Default value: "wysiwyg"
        /// </summary>
        [Category("Configurations")]
        public string StartupMode
        {
            set
            {
                if (value == null) this.Config.Remove("startupMode");
                else this.Config["startupMode"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["startupMode"];
                return c == null ? @"wysiwyg" : FromJsString(c);
            }
        }

        /// <summary>
        /// Whether to automatically enable the "show block" command 
        /// when the editor loads. 
        /// 
        /// Default value: false
        /// </summary>
        [Category("Configurations")]
        public bool StartupOutlineBlocks
        {
            set { this.Config["startupOutlineBlocks"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["startupOutlineBlocks"];
                if (c == null) return false; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The "styles definition set" to use in the editor. 
        /// They will be used in the styles combo and the Style selector of the 
        ///   div container. 
        /// The styles may be defined in the page containing the editor, 
        ///   or can be loaded on demand from an external file. 
        /// In the second case, if this setting contains only a name, 
        ///   the styles definition file will be loaded from the "styles" 
        ///   folder inside the styles plugin folder. 
        /// Otherwise, this setting has the "name:url" syntax, 
        ///   making it possible to set the URL from which loading the styles file.
        /// Previously this setting was available as config.stylesCombo_stylesSet
        /// 
        /// Value type: This is pure javascript, so be careful with it!
        ///   You can easily break that code if you do not escape characters or 
        ///   you misplace an important character.
        ///   If the string starts and ends with [ ], it will pass an array
        ///   Otherwise, it will encapsulate it inside ' '.
        /// 
        /// Default value: 'default'
        /// 
        /// Examples:
        /// // Load from the styles' styles folder (mystyles.js file).
        /// StylesSet = "mystyles";
        /// 
        /// // Load from a relative URL.
        /// StylesSet = "mystyles:/editorstyles/styles.js";
        /// 
        /// // Load from a full URL.
        /// StylesSet = "mystyles:http://www.example.com/editorstyles/styles.js";
        /// 
        /// // Load from a list of definitions.
        /// StylesSet = [
        ///  { name : 'Strong Emphasis', element : 'strong' },
        ///  { name : 'Emphasis', element : 'em' }, ... ];
        /// 
        /// </summary>
        [Category("Configurations")]
        public string StylesSet
        {
            set 
            {
                if (value == null) this.Config.Remove("stylesSet");
                else
                {
                    string val = value.Trim();
                    if (!val.StartsWith("[") || !val.EndsWith("]"))
                    {
                        val = ToJsString(val);
                    }
                    this.Config["stylesSet"] = val;
                }
            }
            get
            {
                string val = this.Config["stylesSet"];
                if (val == null) return @"default";
                if (val.StartsWith("[") && val.EndsWith("]")) return val;
                return FromJsString(val);
            }
        }

        /// <summary>
        /// The editor tabindex value.
        /// 
        /// Default value: 0 (zero)
        /// </summary>
        [Category("Configurations")]
        public int TabIndex
        {
            set { this.Config["tabIndex"] = value.ToString(); }
            get
            {
                string c = this.Config["tabIndex"];
                if (c == null) return 0; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// Instructs the editor to add a number of spaces (&nbsp;) to 
        /// the text when hitting the TAB key. 
        /// If set to zero, the TAB key will be used to move the cursor 
        /// focus to the next element in the page, out of the editor focus. 
        /// 
        /// Default value: 0
        /// </summary>
        [Category("Configurations")]
        public int TabSpaces
        {
            set { this.Config["tabSpaces"] = value.ToString(); }
            get
            {
                string c = this.Config["tabSpaces"];
                if (c == null) return 0; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// The templates definition set to use. 
        /// It accepts a list of names separated by comma. 
        /// It must match definitions loaded with the templates_files setting. 
        /// 
        /// Default value: "default"
        /// </summary>
        [Category("Configurations")]
        public string Templates
        {
            set
            {
                if (value == null) this.Config.Remove("templates");
                else this.Config["templates"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["templates"];
                return c == null ? @"default" : FromJsString(c);
            }
        }

        /// <summary>
        /// The list of templates definition files to load. 
        /// 
        /// Default value: Array of "plugins/templates/templates/default.js"
        /// 
        /// Set to null for default value
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string[] Templates_Files
        {
            set
            {
                if (value != null)
                {
                    StringBuilder sb = new StringBuilder();
                    bool comma = false;
                    sb.Append('[');
                    foreach (string css in value)
                    {
                        if (comma) sb.Append(@","); else comma = true;
                        sb.Append(ToJsString(css));
                    }
                    sb.Append(']');
                    this.Config["templates_files"] = sb.ToString();
                    ViewState["templates_files"] = value;
                }
                else
                {
                    this.Config.Remove("templates_files");
                    ViewState.Remove("templates_files");
                }
            }
            get
            {
                return (string[])ViewState["templates_files"];
            }
        }

        /// <summary>
        /// Whether the "Replace actual contents" checkbox 
        /// is checked by default in the Templates dialog. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool Templates_ReplaceContent
        {
            set { this.Config["templates_replaceContent"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["templates_replaceContent"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The theme to be used to build the UI.
        /// 
        /// Default value: "default"
        /// </summary>
        [Category("Configurations")]
        public string Theme
        {
            set
            {
                if (value == null) this.Config.Remove("theme");
                else this.Config["theme"] = ToJsString(value);
            }
            get
            {
                string c = this.Config["theme"];
                return c == null ? @"default" : FromJsString(c);
            }
        }


        /// <summary>
        /// The toolbox (alias toolbar) definition. 
        /// It is a toolbar name or an array of toolbars (strips), 
        ///   each one being also an array, containing a list of UI items. 
        /// 
        /// Value type: This is pure javascript, so be careful with it!
        ///   You can easily break that code if you do not escape characters or 
        ///   you misplace an important character.
        ///   If the string starts and ends with [ ], it will pass an array
        ///   Otherwise, it will encapsulate it inside ' '.
        /// 
        /// Default value: 'Full'
        /// 
        /// </summary>
        [Category("Configurations")]
        public string Toolbar
        {
            set
            {
                if (value == null) this.Config.Remove("toolbar");
                else
                {
                    string val = value.Trim();
                    if (!val.StartsWith("[") || !val.EndsWith("]"))
                    {
                        val = ToJsString(val);
                    }
                    this.Config["toolbar"] = val;
                }
            }
            get
            {
                string val = this.Config["toolbar"];
                if (val == null) return @"Full";
                if (val.StartsWith("[") && val.EndsWith("]")) return val;
                return FromJsString(val);
            }
        }

        /// <summary>
        /// The toolbar definition. 
        /// It is an array of toolbars (strips), each one being also an array, 
        ///   containing a list of UI items. 
        /// Note that this setting is composed by "toolbar_" added by the 
        ///   toolbar name, which in this case is called "Basic". 
        /// This second part of the setting name can be anything. 
        /// You must use this name in the Toolbar setting, so you instruct the 
        ///   editor which toolbar_(name) setting to you. 
        /// 
        /// Default value: 
        ///  [
        ///   'Bold',
        ///   'Italic',
        ///   '-',
        ///   'NumberedList',
        ///   'BulletedList',
        ///   '-',
        ///   'Link',
        ///   'Unlink',
        ///   '-',
        ///  ];
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Toolbar_Basic
        {
            set
            {
                if (value == null) this.Config.Remove("toolbar_Basic");
                else this.Config["toolbar_Basic"] = value;
            }
            get
            {
                return this.Config["toolbar_Basic"];
            }
        }

        /// <summary>
        /// This is the default toolbar definition used by the editor. It contains all editor features. 
        /// 
        /// Default value: 
        ///  [
        ///   ['Source','-','Save','NewPage','Preview','-','Templates'],
        ///   ['Cut','Copy','Paste','PasteText','PasteFromWord','-','Print', 'SpellChecker', 'Scayt'],
        ///   ['Undo','Redo','-','Find','Replace','-','SelectAll','RemoveFormat'],
        ///   ['Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 'Button', 'ImageButton', 'HiddenField'],
        ///   '/',
        ///   ['Bold','Italic','Underline','Strike','-','Subscript','Superscript'],
        ///   ['NumberedList','BulletedList','-','Outdent','Indent','Blockquote','CreateDiv'],
        ///   ['JustifyLeft','JustifyCenter','JustifyRight','JustifyBlock'],
        ///   ['Link','Unlink','Anchor'],
        ///   ['Image','Flash','Table','HorizontalRule','Smiley','SpecialChar','PageBreak'],
        ///   '/',
        ///   ['Styles','Format','Font','FontSize'],
        ///   ['TextColor','BGColor'],
        ///   ['Maximize', 'ShowBlocks','-','About']
        ///  ];
        /// 
        /// Set to null for default value
        /// 
        /// NOTE: This is a pure javascript value, so be careful not to break things!
        /// </summary>
        /// <returns>null if the default value is set</returns>
        [Category("Configurations")]
        public string Toolbar_Full
        {
            set
            {
                if (value == null) this.Config.Remove("toolbar_Full");
                else this.Config["toolbar_Full"] = value;
            }
            get
            {
                return this.Config["toolbar_Full"];
            }
        }

        /// <summary>
        /// Whether the toolbar can be collapsed by the user. 
        /// If disabled, the collapser button will not be displayed. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool ToolbarCanCollapse
        {
            set { this.Config["toolbarCanCollapse"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["toolbarCanCollapse"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The "theme space" to which rendering the toolbar. 
        /// For the default theme, the recommended options are 
        ///  "top" and "bottom". 
        ///  
        /// Note: I'm not sure yet how is this working. 
        ///       Should I make this a string open to any value? 
        ///       Will anyone ever use values other than "top" and "bottom" 
        ///       With this control?
        /// </summary>
        [Category("Configurations")]
        public ToolbarLocation ToolbarLocation
        {
             set 
             {
                switch (value)
                {
                    default:
                    case ToolbarLocation.Top:
                        this.Config["toolbarLocation"] = "'top'";
                        break;
                    case ToolbarLocation.Bottom:
                        this.Config["toolbarLocation"] = "'bottom'";
                        break;
                }                
            }
            get
            {
                switch (this.Config["toolbarLocation"])
                {
                    default:
                    case "'top'":
                        return ToolbarLocation.Top;
                    case "'bottom'":
                        return ToolbarLocation.Bottom;
                }
            }
        }

        /// <summary>
        /// Whether the toolbar must start expanded when the editor is loaded. 
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool ToolbarStartupExpanded
        {
            set { this.Config["toolbarStartupExpanded"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["toolbarStartupExpanded"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The number of undo steps to be saved. The higher this setting value the more memory is used for it. 
        /// 
        /// Default value: 20
        /// </summary>
        [Category("Configurations")]
        public int UndoStackSize
        {
            set { this.Config["undoStackSize"] = value.ToString(); }
            get
            {
                string c = this.Config["undoStackSize"];
                if (c == null) return 20; // The default value
                return int.Parse(c);
            }
        }

        /// <summary>
        /// Indicates that some of the editor features, like alignement and text direction,
        ///  should used the "computed value" of the feature to indicate it's on/off state,
        ///  instead of using the "real value".
        /// If enabled, in a left to right written document,
        ///  the "Left Justify" alignment button will show as active,
        ///  even if the aligment style is not explicitly applied to the current
        ///  paragraph in the editor.
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool UseComputedState
        {
            set { this.Config["useComputedState"] = (value ? "true" : "false"); }
            get
            {
                string c = this.Config["useComputedState"];
                if (c == null) return true; // The default value
                return bool.Parse(c);
            }
        }

        /// <summary>
        /// The editor width in CSS size format or pixel integer.
        /// 
        /// Default value: empty (Unit.Empty)
        /// 
        /// Set to Unit.Empty to reset to default value
        /// </summary>
        [Category("Configurations")]
        [DefaultValue("")]
        public Unit Width
        {
            set
            {
                if (value.IsEmpty) this.Config.Remove(@"width");
                else this.Config["width"] = ToJsString(value.ToString());
            }
            get 
            {
                string c = this.Config["width"];
                if (c == null) return Unit.Empty;
                return Unit.Parse(FromJsString(c));
            }
        }

        /// <summary>
        /// Sets the UI color, using the 'uicolor' plugin.
        /// Works only for Kama skin.
        /// 
        /// This property registers the UIColor plugin.
        /// 
        /// To remove - set to Color.Transparent
        /// </summary>
        [Category("Appearance")]
        [TypeConverter(typeof(WebColorConverter))]
        public Color UIColor
        {
            set
            {
                if (value == Color.Transparent)
                {
                    this.Config.Remove("uiColor");
                    RemoveExtraPlugin("uicolor");
                }
                else
                {
                    if (UIColor == Color.Transparent) AddExtraPlugin("uicolor"); // Do not add twice
                    this.Config["uiColor"] = ToJsString("#" + value.R.ToString(@"x2") + value.G.ToString(@"x2") + value.B.ToString(@"x2"));
                }
            }
            get
            {
                if (this.Config["uiColor"] == null) return Color.Transparent;
                return System.Drawing.ColorTranslator.FromHtml(FromJsString(this.Config["uiColor"]));
            }
        }

		#endregion

        #region Other "custom" configurations

        /// <summary>
        /// Whether the replaced element (usually a textarea) is to be 
        /// updated automatically when posting the form containing the editor.
        /// 
        /// Default value: true
        /// </summary>
        [Category("Configurations")]
        public bool RegisterPlugin_TableResize
        {
            set 
            { 
                if (value)
                {
                    AddExtraPlugin(@"tableresize");
                }
                else
                {
                    RemoveExtraPlugin(@"tableresize");
                }
            }
            get
            {
                string str = ExtraPlugins_List.Find(delegate(string val) { return val == @"tableresize"; });
                return str != null;
            }
        }

        #endregion

        #region Utility

        private static string ToJsString(string unescaped)
        {
            if (unescaped == null) return null;
            return '\'' + unescaped.Replace(@"\", @"\\").Replace(@"'", @"\'").Replace("\n", @"\n").Replace("\r", @"\r") + '\'';
        }

        private static string FromJsString(string escaped)
        {
            if (escaped == null) return null;
            escaped = escaped.Trim();
            if (escaped.StartsWith("'") && escaped.EndsWith(@"'"))
            {
                escaped = escaped.Substring(1, escaped.Length - 2);
            }
            return escaped.Replace(@"\r", "\r").Replace(@"\n", "\n").Replace(@"\'", @"'").Replace(@"\\", @"\");
        }

        protected string GetControlClientID(string name)
        {
            Control c = this.FindControl(name);
            if (c == null) return string.Empty;
            return c.ClientID;
        }

        protected string SeparateAttributePairs(string[] attributes)
        {
            StringBuilder sb = new StringBuilder();
            bool comma = false;
            int idxSep;
            foreach (string attr in attributes)
            {
                idxSep = attr.IndexOf(':');
                if (idxSep > -1)
                {
                    if (comma) sb.Append(','); else comma = true;
                    sb.Append(ToJsString(attr.Substring(0, idxSep)));
                    sb.Append(':');
                    sb.Append(ToJsString(attr.Substring(idxSep + 1)));
                }
            }
            return sb.ToString();
        }

        protected void PrepareSharedSpacesConfig()
        {
            string top = SharedSpacesTopClientID;
            if (top.Length == 0) 
            {
                top = SharedSpacesTop;
                if (top.Length > 0) top = GetControlClientID(top);
            }
            string bottom = SharedSpacesBottomClientID;
            if (bottom.Length == 0)
            {
                bottom = SharedSpacesBottom;
                if (bottom.Length > 0) bottom = GetControlClientID(bottom);
            }
            if (top.Length > 0 || bottom.Length > 0)
            {
                string config = "{";
                if (top.Length > 0)
                {
                    config += @"top:'" + top + @"'";
                    // Removes the maximize plugin as it's not usable in a shared toolbar.
                    // This may cause duplicate values if the user has already specified this, but it does not matter at all
                    RemovePlugin(@"maximize"); 
                }
                if (bottom.Length>0)
                {
                    if (top.Length > 0) config += ',';
                    config += @"bottom:'" + bottom + @"'";
                    // Removes the resizer as it's not usable in a shared elements path.
                    // This may cause duplicate values if the user has already specified this, but it does not matter at all
                    RemovePlugin(@"resize");
                }
                config += @"}";
                this.Config[@"sharedSpaces"] = config;
            }
        }

        /// <summary>
        /// Do stuff that is required to prepare certain config entries,
        /// just before rendering
        /// </summary>
        protected void ConfigPreRender()
        {
            // Prepare shared spaces config
            PrepareSharedSpacesConfig(); 
            // Refresh config items from their equivalent arrays if needed
            UpdateConfig_ColorButton_Colors();
            UpdateConfig_RemovePlugins();
            UpdateConfig_ExtraPlugins();
        }

        #endregion

        #region Rendering

        protected override void OnInit(EventArgs e)
        {
            this._IsCompatible = this.CheckBrowserCompatibility();
            if (HasMsAjax && this._IsCompatible) Page.ClientScript.RegisterClientScriptInclude(this.GetType(), @"CKEditor.js", ResolveUrl(BasePath + (BasePath.EndsWith(@"/") ? "" : "/") + "ckeditor.js"));
        }

        protected override void OnPreRender(EventArgs e)
        {
            base.OnPreRender(e);
            if (this._IsCompatible)
            {
                // I'm using RegisterClientScriptInclude and not direct script
                // injection because I don't want the js to be included twice.
                // If it is included twice the CKEditor commits suicide... 
                // Maybe this is a bug in ckeditor, dunno, doesn't matter.

                //RegisterClientScriptInclude(@"CKEditor.js", ResolveUrl(BasePath + (BasePath.EndsWith(@"/") ? "" : "/") + "ckeditor.js"));
                if (!HasMsAjax) Page.ClientScript.RegisterClientScriptInclude(this.GetType(), @"CKEditor.js", ResolveUrl(BasePath + (BasePath.EndsWith(@"/") ? "" : "/") + "ckeditor.js"));

                if (HasMsAjax)
                {
                    // If there may be an AJAX postback, the form.onSubmit is not called.
                    // So we need to 'register' a postback client script,
                    // In order to update the value in the TEXTAREA.
                    // Then we have to remove the CKEditor for cleanup, 
                    // because there's a bug when doing the cleanup if the CKEditor do not exist anymore.

                    string postbackScript;
                    if (HtmlEncodeOutput)
                    {
                        postbackScript = string.Format(
                             @"if (CKEDITOR && CKEDITOR.instances && CKEDITOR.instances.{0}) {{var e=document.getElementById('{0}'); if(e) {{var i=CKEDITOR.instances.{0}; e.value=i.getData().replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');i.fire('destroy'); }}}};",
                             this.ClientID);}
                    else
                    {
                        postbackScript = string.Format(
                             @"if (CKEDITOR && CKEDITOR.instances && CKEDITOR.instances.{0}) {{var e=document.getElementById('{0}'); if(e) {{var i=CKEDITOR.instances.{0}; e.value=i.getData();i.fire('destroy'); }}}};",
                             this.ClientID);
                    }
                    RegisterOnSubmitStatement(
                        this.GetType(), 
                        this.ClientID + @"_OnClientPostback",
                        postbackScript);

                    postbackScript =
                        string.Format(@"document.getElementById('{0}').dispose=function(){{if (CKEDITOR && CKEDITOR.instances && CKEDITOR.instances.{0}) {{var e=document.getElementById('{0}'); if(e) {{var i=CKEDITOR.instances.{0}; CKEDITOR.remove(i);}}}}}};", this.ClientID);
                    RegisterStartupScript(
                        this.GetType(),
                        this.ClientID + @"_Dispose",
                        postbackScript, 
                        true);
                }

                if (_IsCompatible)
                {
                    ConfigPreRender();

                    StringBuilder sbScript = new StringBuilder();

                    if (HasMsAjax)
                    {
                        sbScript.AppendFormat(@"if (CKEDITOR && CKEDITOR.instances && CKEDITOR.instances.{0}) {{CKEDITOR.instances.{0}.destroy();}}", this.ClientID);
                    }

                    sbScript.AppendFormat(@"CKEDITOR.replace( '{0}', {{", this.ClientID);
                    sbScript.Append(this.Config.GetJsConfigArray());
                    sbScript.Append(@"});");

                    /* // A fix for CKEditor 3.0.0 - 3.0.2 where there's no htmlEncodeOutput
                    if (this.HtmlEncodeOutput)
                    {
                        sbScript.Append(
                        @"
                        CKEDITOR.instances.{0}.updateElement = CKEDITOR.tools.override(CKEDITOR.instances.{0}.updateElement, function(originalUpdateElement) {
                        return function(){
                            var thisObj = this;
                            var oldGetData = this.getData;
                            this.getData = function(){ return oldGetData.call(thisObj).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;'); };
                            originalUpdateElement.call(thisObj);
                            this.getData = oldGetData;
                            }
                        });
                        ", this.ClientID
                        );
                    }
                    */

                    RegisterStartupScript(this.GetType(), this.ClientID + @"_CKE_Startup", sbScript.ToString(), true);
                }
            }
        }

		protected override void Render(HtmlTextWriter writer)
		{
            writer.Write(
            "<textarea name=\"{0}\" id=\"{1}\" rows=\"4\" cols=\"40\" style=\"width: {2}; height: {3}\" wrap=\"virtual\">{4}</textarea>",
                this.UniqueID,
                this.ClientID,
                this.Width,
                this.Height,
                System.Web.HttpUtility.HtmlEncode(this.Value));
		}

		#endregion

        #region Microsoft Ajax (Update Panels...)
        private static bool? _HasMsAjax = null;
        private static Type typScriptManager;
        private static System.Reflection.MethodInfo mtdRegisterClientScriptInclude;
        private static System.Reflection.MethodInfo mtdRegisterStartupScript;
        private static System.Reflection.MethodInfo mtdRegisterOnSubmitStatement;

        /// <summary>
        /// This figures out if Microsoft Ajax is enabled
        /// </summary>
        private bool HasMsAjax
        {
            get
            {
                if (_HasMsAjax == null)
                {
                    _HasMsAjax = false;
                    System.Reflection.Assembly[] AppAssemblies = AppDomain.CurrentDomain.GetAssemblies();
                    foreach (System.Reflection.Assembly asm in AppAssemblies)
                    {
                        if (asm.ManifestModule.Name == "System.Web.Extensions.dll")
                        {
                            try
                            {
                                typScriptManager = asm.GetType("System.Web.UI.ScriptManager");
                                if (typScriptManager != null) _HasMsAjax = true;
                            }
                            catch { }
                            break;
                        }
                    }
                }

                return _HasMsAjax ?? false;
            }
        }

        /// <summary>
        /// This registers a javascript include with compatibility with the Microsoft Ajax
        /// </summary>
        private void RegisterClientScriptInclude(Type type, string key, string url)
        {
            if (HasMsAjax)
            {
                if (mtdRegisterClientScriptInclude == null)
                {
                    mtdRegisterClientScriptInclude = typScriptManager.GetMethod("RegisterClientScriptInclude", new Type[4] { typeof(Control), typeof(Type), typeof(string), typeof(string) });
                }

                mtdRegisterClientScriptInclude.Invoke(null, new object[4] { this, type, key, url });
            }
            else
            {
                Page.ClientScript.RegisterClientScriptInclude(type, key, url);
            }
        }

        /// <summary>
        /// This registers a startup javascript with compatibility with the Microsoft Ajax
        /// </summary>
        private void RegisterStartupScript(Type type, string key, string script, bool addScriptTags)
        {
            if (HasMsAjax)
            {
                if (mtdRegisterStartupScript == null)
                {
                    mtdRegisterStartupScript = typScriptManager.GetMethod("RegisterStartupScript", new Type[5] { typeof(Control), typeof(Type), typeof(string), typeof(string), typeof(bool) });
                }

                mtdRegisterStartupScript.Invoke(null, new object[5] { this, type, key, script, addScriptTags });
            }
            else
            {
                Page.ClientScript.RegisterStartupScript(type, key, script, true);
            }
        }

        /// <summary>
        /// This registers a startup javascript with compatibility with the Microsoft Ajax
        /// </summary>
        private void RegisterOnSubmitStatement(Type type, string key, string script)
        {
            if (HasMsAjax)
            {
                if (mtdRegisterOnSubmitStatement == null)
                {
                    mtdRegisterOnSubmitStatement = typScriptManager.GetMethod("RegisterOnSubmitStatement", new Type[4] { typeof(Control), typeof(Type), typeof(string), typeof(string) });
                }

                mtdRegisterOnSubmitStatement.Invoke(null, new object[4] { this, type, key, script });
            }
            else
            {
                Page.ClientScript.RegisterOnSubmitStatement(type, key, script);
            }
        }

        #endregion

        #region Compatibility Check

        public bool CheckBrowserCompatibility()
		{
			return IsCompatibleBrowser();
		}

		/// <summary>
		/// Checks if the current HTTP request comes from a browser compatible
        /// with CKEditor.
		/// </summary>
		/// <returns>"true" if the browser is compatible.</returns>
		public static bool IsCompatibleBrowser()
		{
			return IsCompatibleBrowser( System.Web.HttpContext.Current.Request );
		}

		/// <summary>
		/// Checks if the provided HttpRequest object comes from a browser
        /// compatible with CKEditor.
		/// </summary>
		/// <returns>"true" if the browser is compatible.</returns>
		public static bool IsCompatibleBrowser( System.Web.HttpRequest request )
		{
			System.Web.HttpBrowserCapabilities oBrowser = request.Browser;

			// Internet Explorer 6.0+ for Windows
			if ( oBrowser.Browser == "IE" && ( oBrowser.MajorVersion >= 6 ) && oBrowser.Win32 )
				return true;

			string sUserAgent = request.UserAgent;

			if ( sUserAgent.IndexOf( "Gecko/" ) >= 0 )
			{
				Match oMatch = Regex.Match( request.UserAgent, @"(?<=Gecko/)\d{8}" );
				return ( oMatch.Success && int.Parse( oMatch.Value, CultureInfo.InvariantCulture ) >= 20030210 );
			}

			if ( sUserAgent.IndexOf( "Opera/" ) >= 0 )
			{
				Match oMatch = Regex.Match( request.UserAgent, @"(?<=Opera/)[\d\.]+" );
				return ( oMatch.Success && float.Parse( oMatch.Value, CultureInfo.InvariantCulture ) >= 9.5 );
			}

			if ( sUserAgent.IndexOf( "AppleWebKit/" ) >= 0 )
			{
				Match oMatch = Regex.Match( request.UserAgent, @"(?<=AppleWebKit/)\d+" );
				return ( oMatch.Success && int.Parse( oMatch.Value, CultureInfo.InvariantCulture ) >= 522 );
			}

			return false;
		}

		#endregion

		#region Postback Handling

        public virtual bool LoadPostData(string postDataKey, System.Collections.Specialized.NameValueCollection postCollection)
		{
            if (HtmlEncodeOutput)
            {
                string postedValue = postCollection[postDataKey];

                // Revert the Html encoding.
                postedValue = postedValue.Replace("&lt;", "<");
                postedValue = postedValue.Replace("&gt;", ">");
                postedValue = postedValue.Replace("&amp;", "&");

                if (postedValue != this.Value)
                {
                    this.Value = postedValue;
                    return true;
                }
            }
			return false ;
		}

		public virtual void RaisePostDataChangedEvent()
		{
			// Do nothing
		}

		#endregion
	}
}
